# -*- coding: utf-8 -*-
# Part of Odoo. See LICENSE file for full copyright and licensing details.
from contextlib import contextmanager
import json
import requests
from unittest.mock import Mock, patch

from odoo.addons.delivery_easypost.models.delivery_carrier import DeliverCarrier
from odoo.addons.delivery_easypost.models.easypost_request import EasypostRequest
from odoo.tools import file_open
from odoo.tests import TransactionCase


class EasypostTestCommon(TransactionCase):
    def setUp(self):
        super().setUp()
        self.your_company = self.env.ref("base.main_partner")
        self.your_company.write(
            {
                "name": "Odoo SA",
                "street": "44 Wall Street",
                "street2": "Suite 603",
                "city": "New York",
                "zip": 10005,
                "state_id": self.env.ref("base.state_us_27").id,
                "country_id": self.env.ref("base.us").id,
                "phone": "+1 (929) 352-6366",
            }
        )

        self.jackson = self.env.ref("base.res_partner_10")
        self.jackson.write(
            {
                "street": "1515 Main Street",
                "street2": "",
                "city": "Columbia",
                "zip": 29201,
                "state_id": self.env.ref("base.state_us_41").id,
                "country_id": self.env.ref("base.us").id,
            }
        )
        self.agrolait = self.env.ref("base.res_partner_2")
        self.agrolait.write(
            {
                "country_id": self.env.ref("base.be").id,
                "city": "Auderghem-Ouderghem",
                "street": "Avenue Edmond Van Nieuwenhuyse",
                "zip": "1160",
            }
        )
        # configure rounding, so that we can enter an extra-light product
        conf = self.env["ir.config_parameter"]
        conf.set_param("product.weight_in_lbs", "1")
        precision = self.env.ref("product.decimal_stock_weight")
        precision.digits = 4
        self.uom_lbs = self.env.ref("uom.product_uom_lb")
        self.uom_lbs.rounding = 0.0001
        self.server = self.env["product.product"].create(
            {
                "name": "server",
                "hs_code": "8471.49.00.00",
                "type": "consu",
                "weight": 3.0,
                "volume": 4.0,
            }
        )
        self.miniServer = self.env["product.product"].create(
            {
                "name": "mini server",
                "hs_code": "8471.49.00.00",
                "type": "consu",
                "weight": 2.0,
                "volume": 0.35,
            }
        )
        self.microServer = self.env["product.product"].create(
            {
                "name": "micro server",
                "type": "consu",
                "weight": 0.0025,
                "volume": 0.35,
            }
        )

        self.easypost_fedex_carrier_product = self.env["product.product"].create(
            {
                "name": "Fedex Easypost",
                "type": "service",
            }
        )

        self.easypost_fedex_carrier = self.env["delivery.carrier"].create(
            {
                "name": "EASYPOST FedEx",
                "delivery_type": "easypost",
                "easypost_test_api_key": "EZTKc116818834b24215a47fafc556e46340LYwlOtD5xn1m1euw0HmZ5A",
                "easypost_production_api_key": "zhiDnLnzKCVkelNzVAfWEQ",
                "product_id": self.easypost_fedex_carrier_product.id,
            }
        )
        product_type_wizard = self.easypost_fedex_carrier.action_get_carrier_type()
        self.easypost_fedex_carrier.easypost_delivery_type = "FedEx"
        self.easypost_fedex_carrier.easypost_delivery_type_id = product_type_wizard["context"]["carrier_types"]["FedEx"]

        self.fedex_default_package_type = self.env["stock.package.type"].create(
            {
                "name": "My FedEx Box",
                "package_carrier_type": "easypost",
                "max_weight": 10,
                "height": 10,
                "packaging_length": 10,
                "width": 10,
            }
        )
        self.easypost_fedex_carrier.easypost_default_package_type_id = self.fedex_default_package_type

    @contextmanager
    def patch_easypost_requests(self):
        """ Mock context for get and post requests to the Easypost API. """
        def _mocked_easypost_request(endpoint, request_type='get', data=None):
            mocked_contents = {
                'post': {
                    'orders': {'mode': 'test', 'reference': 'odoo', 'is_return': False, 'options': {'currency': 'USD', 'payment': {'type': 'SENDER'}}, 'messages': [], 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'customs_info': None, 'from_address': {'id': 'adr_c6ae26be343c11ed9445ac1f6bc7b362', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'Odoo SA', 'company': 'Odoo SA', 'street1': '44 Wall Street', 'street2': 'Suite 603', 'city': 'New York', 'state': 'NY', 'zip': '10005', 'country': 'US', 'phone': '19293526366', 'email': 'info@yourcompany.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'to_address': {'id': 'adr_c6ac8a1f343c11edbe5aac1f6bc72124', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'The Jackson Group', 'company': 'The Jackson Group', 'street1': '1515 Main Street', 'street2': None, 'city': 'Columbia', 'state': 'SC', 'zip': '29201', 'country': 'US', 'phone': '3345021024', 'email': 'jackson.group82@example.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'buyer_address': {'id': 'adr_c6ac8a1f343c11edbe5aac1f6bc72124', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'The Jackson Group', 'company': 'The Jackson Group', 'street1': '1515 Main Street', 'street2': None, 'city': 'Columbia', 'state': 'SC', 'zip': '29201', 'country': 'US', 'phone': '3345021024', 'email': 'jackson.group82@example.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'return_address': {'id': 'adr_c6ae26be343c11ed9445ac1f6bc7b362', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'Odoo SA', 'company': 'Odoo SA', 'street1': '44 Wall Street', 'street2': 'Suite 603', 'city': 'New York', 'state': 'NY', 'zip': '10005', 'country': 'US', 'phone': '19293526366', 'email': 'info@yourcompany.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'shipments': [{'created_at': '2022-09-14T14:51:55Z', 'is_return': False, 'messages': [], 'mode': 'test', 'options': {'label_format': 'PDF', 'label_date': '2022-09-14T14:51:54+00:00', 'currency': 'USD', 'payment': {'type': 'SENDER'}, 'date_advance': 0}, 'reference': None, 'status': 'unknown', 'tracking_code': None, 'updated_at': '2022-09-14T14:51:55Z', 'batch_id': None, 'batch_status': None, 'batch_message': None, 'customs_info': {'id': 'cstinfo_efd5b483865347449b8a831c469df0f2', 'object': 'CustomsInfo', 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'contents_explanation': None, 'contents_type': 'merchandise', 'customs_certify': False, 'customs_signer': None, 'eel_pfc': None, 'non_delivery_option': 'return', 'restriction_comments': None, 'restriction_type': 'none', 'mode': 'test', 'declaration': None, 'customs_items': [{'id': 'cstitem_3c169b67bc004723a1125dae225276ae', 'object': 'CustomsItem', 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'description': 'server', 'hs_tariff_number': 'False', 'origin_country': 'US', 'quantity': 1, 'value': '1.15', 'weight': 48.0, 'code': None, 'mode': 'test', 'manufacturer': None, 'currency': 'USD', 'eccn': None, 'printed_commodity_identifier': None}, {'id': 'cstitem_61e4b578b7ef4bf7846582826e67afbc', 'object': 'CustomsItem', 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'description': 'mini server', 'hs_tariff_number': 'False', 'origin_country': 'US', 'quantity': 1, 'value': '1.15', 'weight': 32.0, 'code': None, 'mode': 'test', 'manufacturer': None, 'currency': 'USD', 'eccn': None, 'printed_commodity_identifier': None}]}, 'from_address': {'id': 'adr_c6ae26be343c11ed9445ac1f6bc7b362', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'Odoo SA', 'company': 'Odoo SA', 'street1': '44 Wall Street', 'street2': 'Suite 603', 'city': 'New York', 'state': 'NY', 'zip': '10005', 'country': 'US', 'phone': '19293526366', 'email': 'info@yourcompany.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'insurance': None, 'order_id': 'order_7bf14e4a887c42d7b41f06a79bdf13ea', 'parcel': {'id': 'prcl_342dd1481f9c4c27adc44500478b70bf', 'object': 'Parcel', 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'length': 10.0, 'width': 10.0, 'height': 10.0, 'predefined_package': None, 'weight': 80.0, 'mode': 'test'}, 'postage_label': None, 'rates': [{'id': 'rate_31dc028a46d04741b934fd25d082a208', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FIRST_OVERNIGHT', 'carrier': 'FedEx', 'rate': '173.51', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '173.51', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T08:00:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_ca04097ed6aa4b7babe0a1499b6b6914', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'PRIORITY_OVERNIGHT', 'carrier': 'FedEx', 'rate': '136.15', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '136.15', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T10:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_5d4e753402af4a7a86e5b180fd916111', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'STANDARD_OVERNIGHT', 'carrier': 'FedEx', 'rate': '128.81', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '128.81', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_a17f804119484ec3b9607a284e6834e7', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FEDEX_2_DAY_AM', 'carrier': 'FedEx', 'rate': '52.28', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '52.28', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T10:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_4e186801ce314b03aa9480cde74cc91f', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FEDEX_2_DAY', 'carrier': 'FedEx', 'rate': '45.05', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '45.05', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_19c859344a4a4f17a8997c9ae8ff9f8f', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FEDEX_GROUND', 'carrier': 'FedEx', 'rate': '17.27', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '17.27', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T23:59:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_7afecdaa1a924b92837827f95e4e70d6', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FEDEX_EXPRESS_SAVER', 'carrier': 'FedEx', 'rate': '40.63', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '40.63', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 5, 'delivery_date': '2022-09-19T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 5, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}], 'refund_status': None, 'scan_form': None, 'selected_rate': None, 'tracker': None, 'to_address': {'id': 'adr_c6ac8a1f343c11edbe5aac1f6bc72124', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'The Jackson Group', 'company': 'The Jackson Group', 'street1': '1515 Main Street', 'street2': None, 'city': 'Columbia', 'state': 'SC', 'zip': '29201', 'country': 'US', 'phone': '3345021024', 'email': 'jackson.group82@example.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'usps_zone': 4, 'return_address': {'id': 'adr_c6ae26be343c11ed9445ac1f6bc7b362', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'Odoo SA', 'company': 'Odoo SA', 'street1': '44 Wall Street', 'street2': 'Suite 603', 'city': 'New York', 'state': 'NY', 'zip': '10005', 'country': 'US', 'phone': '19293526366', 'email': 'info@yourcompany.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'buyer_address': {'id': 'adr_c6ac8a1f343c11edbe5aac1f6bc72124', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'The Jackson Group', 'company': 'The Jackson Group', 'street1': '1515 Main Street', 'street2': None, 'city': 'Columbia', 'state': 'SC', 'zip': '29201', 'country': 'US', 'phone': '3345021024', 'email': 'jackson.group82@example.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'forms': [], 'fees': [], 'id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'object': 'Shipment'}], 'rates': [{'id': 'rate_31dc028a46d04741b934fd25d082a208', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FIRST_OVERNIGHT', 'carrier': 'FedEx', 'rate': '173.51', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '173.51', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T08:00:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_ca04097ed6aa4b7babe0a1499b6b6914', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'PRIORITY_OVERNIGHT', 'carrier': 'FedEx', 'rate': '136.15', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '136.15', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T10:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_5d4e753402af4a7a86e5b180fd916111', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'STANDARD_OVERNIGHT', 'carrier': 'FedEx', 'rate': '128.81', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '128.81', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_a17f804119484ec3b9607a284e6834e7', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FEDEX_2_DAY_AM', 'carrier': 'FedEx', 'rate': '52.28', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '52.28', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T10:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_4e186801ce314b03aa9480cde74cc91f', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FEDEX_2_DAY', 'carrier': 'FedEx', 'rate': '45.05', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '45.05', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_19c859344a4a4f17a8997c9ae8ff9f8f', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FEDEX_GROUND', 'carrier': 'FedEx', 'rate': '17.27', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '17.27', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T23:59:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_7afecdaa1a924b92837827f95e4e70d6', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FEDEX_EXPRESS_SAVER', 'carrier': 'FedEx', 'rate': '40.63', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '40.63', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 5, 'delivery_date': '2022-09-19T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 5, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}], 'id': 'order_7bf14e4a887c42d7b41f06a79bdf13ea', 'object': 'Order'},
                    'orders/order_7bf14e4a887c42d7b41f06a79bdf13ea/buy': {'mode': 'test', 'reference': 'odoo', 'is_return': False, 'options': {'currency': 'USD', 'payment': {'type': 'SENDER'}}, 'messages': [], 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'customs_info': None, 'from_address': {'id': 'adr_c6ae26be343c11ed9445ac1f6bc7b362', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'Odoo SA', 'company': 'Odoo SA', 'street1': '44 Wall Street', 'street2': 'Suite 603', 'city': 'New York', 'state': 'NY', 'zip': '10005', 'country': 'US', 'phone': '19293526366', 'email': 'info@yourcompany.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'to_address': {'id': 'adr_c6ac8a1f343c11edbe5aac1f6bc72124', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'The Jackson Group', 'company': 'The Jackson Group', 'street1': '1515 Main Street', 'street2': None, 'city': 'Columbia', 'state': 'SC', 'zip': '29201', 'country': 'US', 'phone': '3345021024', 'email': 'jackson.group82@example.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'buyer_address': {'id': 'adr_c6ac8a1f343c11edbe5aac1f6bc72124', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'The Jackson Group', 'company': 'The Jackson Group', 'street1': '1515 Main Street', 'street2': None, 'city': 'Columbia', 'state': 'SC', 'zip': '29201', 'country': 'US', 'phone': '3345021024', 'email': 'jackson.group82@example.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'return_address': {'id': 'adr_c6ae26be343c11ed9445ac1f6bc7b362', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'Odoo SA', 'company': 'Odoo SA', 'street1': '44 Wall Street', 'street2': 'Suite 603', 'city': 'New York', 'state': 'NY', 'zip': '10005', 'country': 'US', 'phone': '19293526366', 'email': 'info@yourcompany.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'shipments': [{'created_at': '2022-09-14T14:51:55Z', 'is_return': False, 'messages': [], 'mode': 'test', 'options': {'label_format': 'PDF', 'label_date': '2022-09-14T14:51:54+00:00', 'currency': 'USD', 'payment': {'type': 'SENDER'}, 'date_advance': 0}, 'reference': None, 'status': 'unknown', 'tracking_code': '794666717383', 'updated_at': '2022-09-14T14:51:58Z', 'batch_id': None, 'batch_status': None, 'batch_message': None, 'customs_info': {'id': 'cstinfo_efd5b483865347449b8a831c469df0f2', 'object': 'CustomsInfo', 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'contents_explanation': None, 'contents_type': 'merchandise', 'customs_certify': False, 'customs_signer': None, 'eel_pfc': None, 'non_delivery_option': 'return', 'restriction_comments': None, 'restriction_type': 'none', 'mode': 'test', 'declaration': None, 'customs_items': [{'id': 'cstitem_3c169b67bc004723a1125dae225276ae', 'object': 'CustomsItem', 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'description': 'server', 'hs_tariff_number': 'False', 'origin_country': 'US', 'quantity': 1, 'value': '1.15', 'weight': 48.0, 'code': None, 'mode': 'test', 'manufacturer': None, 'currency': 'USD', 'eccn': None, 'printed_commodity_identifier': None}, {'id': 'cstitem_61e4b578b7ef4bf7846582826e67afbc', 'object': 'CustomsItem', 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'description': 'mini server', 'hs_tariff_number': 'False', 'origin_country': 'US', 'quantity': 1, 'value': '1.15', 'weight': 32.0, 'code': None, 'mode': 'test', 'manufacturer': None, 'currency': 'USD', 'eccn': None, 'printed_commodity_identifier': None}]}, 'from_address': {'id': 'adr_c6ae26be343c11ed9445ac1f6bc7b362', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'Odoo SA', 'company': 'Odoo SA', 'street1': '44 Wall Street', 'street2': 'Suite 603', 'city': 'New York', 'state': 'NY', 'zip': '10005', 'country': 'US', 'phone': '19293526366', 'email': 'info@yourcompany.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'insurance': None, 'order_id': 'order_7bf14e4a887c42d7b41f06a79bdf13ea', 'parcel': {'id': 'prcl_342dd1481f9c4c27adc44500478b70bf', 'object': 'Parcel', 'created_at': '2022-09-14T14:51:55Z', 'updated_at': '2022-09-14T14:51:55Z', 'length': 10.0, 'width': 10.0, 'height': 10.0, 'predefined_package': None, 'weight': 80.0, 'mode': 'test'}, 'postage_label': {'object': 'PostageLabel', 'id': 'pl_06d5ef89dd0d4aa380c4d4f9f96c4b26', 'created_at': '2022-09-14T14:51:58Z', 'updated_at': '2022-09-14T14:51:58Z', 'date_advance': 0, 'integrated_form': 'none', 'label_date': '2022-09-14T14:51:54Z', 'label_resolution': 200, 'label_size': 'PAPER_4X6', 'label_type': 'default', 'label_file_type': 'application/pdf', 'label_url': 'https://easypost-files.s3.us-west-2.amazonaws.com/files/postage_label/20220914/5d74235b5ccc4de7a596600427f07439.pdf', 'label_pdf_url': 'https://easypost-files.s3.us-west-2.amazonaws.com/files/postage_label/20220914/5d74235b5ccc4de7a596600427f07439.pdf', 'label_zpl_url': None, 'label_epl2_url': None, 'label_file': None}, 'rates': [{'id': 'rate_31dc028a46d04741b934fd25d082a208', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FIRST_OVERNIGHT', 'carrier': 'FedEx', 'rate': '173.51', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '173.51', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T08:00:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_ca04097ed6aa4b7babe0a1499b6b6914', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'PRIORITY_OVERNIGHT', 'carrier': 'FedEx', 'rate': '136.15', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '136.15', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T10:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_5d4e753402af4a7a86e5b180fd916111', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'STANDARD_OVERNIGHT', 'carrier': 'FedEx', 'rate': '128.81', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '128.81', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_a17f804119484ec3b9607a284e6834e7', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FEDEX_2_DAY_AM', 'carrier': 'FedEx', 'rate': '52.28', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '52.28', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T10:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_4e186801ce314b03aa9480cde74cc91f', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FEDEX_2_DAY', 'carrier': 'FedEx', 'rate': '45.05', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '45.05', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_19c859344a4a4f17a8997c9ae8ff9f8f', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FEDEX_GROUND', 'carrier': 'FedEx', 'rate': '17.27', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '17.27', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T23:59:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_7afecdaa1a924b92837827f95e4e70d6', 'object': 'Rate', 'created_at': '2022-09-14T14:51:56Z', 'updated_at': '2022-09-14T14:51:56Z', 'mode': 'test', 'service': 'FEDEX_EXPRESS_SAVER', 'carrier': 'FedEx', 'rate': '40.63', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '40.63', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 5, 'delivery_date': '2022-09-19T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 5, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}], 'refund_status': None, 'scan_form': None, 'selected_rate': {'id': 'rate_5d4e753402af4a7a86e5b180fd916111', 'object': 'Rate', 'created_at': '2022-09-14T14:51:58Z', 'updated_at': '2022-09-14T14:51:58Z', 'mode': 'test', 'service': 'STANDARD_OVERNIGHT', 'carrier': 'FedEx', 'rate': '128.81', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '128.81', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, 'tracker': {'id': 'trk_53403ad6142c4d8a89b1d5b55f0cb1c6', 'object': 'Tracker', 'mode': 'test', 'tracking_code': '794666717383', 'status': 'unknown', 'status_detail': 'unknown', 'created_at': '2022-09-14T14:51:58Z', 'updated_at': '2022-09-14T14:51:58Z', 'signed_by': None, 'weight': None, 'est_delivery_date': None, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier': 'FedEx', 'tracking_details': [], 'fees': [], 'carrier_detail': None, 'public_url': 'https://track.easypost.com/djE6dHJrXzUzNDAzYWQ2MTQyYzRkOGE4OWIxZDViNTVmMGNiMWM2'}, 'to_address': {'id': 'adr_c6ac8a1f343c11edbe5aac1f6bc72124', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'The Jackson Group', 'company': 'The Jackson Group', 'street1': '1515 Main Street', 'street2': None, 'city': 'Columbia', 'state': 'SC', 'zip': '29201', 'country': 'US', 'phone': '3345021024', 'email': 'jackson.group82@example.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'usps_zone': 4, 'return_address': {'id': 'adr_c6ae26be343c11ed9445ac1f6bc7b362', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'Odoo SA', 'company': 'Odoo SA', 'street1': '44 Wall Street', 'street2': 'Suite 603', 'city': 'New York', 'state': 'NY', 'zip': '10005', 'country': 'US', 'phone': '19293526366', 'email': 'info@yourcompany.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'buyer_address': {'id': 'adr_c6ac8a1f343c11edbe5aac1f6bc72124', 'object': 'Address', 'created_at': '2022-09-14T14:51:55+00:00', 'updated_at': '2022-09-14T14:51:55+00:00', 'name': 'The Jackson Group', 'company': 'The Jackson Group', 'street1': '1515 Main Street', 'street2': None, 'city': 'Columbia', 'state': 'SC', 'zip': '29201', 'country': 'US', 'phone': '3345021024', 'email': 'jackson.group82@example.com', 'mode': 'test', 'carrier_facility': None, 'residential': None, 'federal_tax_id': None, 'state_tax_id': None, 'verifications': {}}, 'forms': [], 'fees': [{'object': 'Fee', 'type': 'LabelFee', 'amount': '0.00000', 'charged': True, 'refunded': False}], 'id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'object': 'Shipment'}], 'rates': [{'id': 'rate_31dc028a46d04741b934fd25d082a208', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FIRST_OVERNIGHT', 'carrier': 'FedEx', 'rate': '173.51', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '173.51', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T08:00:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_ca04097ed6aa4b7babe0a1499b6b6914', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'PRIORITY_OVERNIGHT', 'carrier': 'FedEx', 'rate': '136.15', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '136.15', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T10:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_5d4e753402af4a7a86e5b180fd916111', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'STANDARD_OVERNIGHT', 'carrier': 'FedEx', 'rate': '128.81', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '128.81', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 1, 'delivery_date': '2022-09-15T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 1, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_a17f804119484ec3b9607a284e6834e7', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FEDEX_2_DAY_AM', 'carrier': 'FedEx', 'rate': '52.28', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '52.28', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T10:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_4e186801ce314b03aa9480cde74cc91f', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FEDEX_2_DAY', 'carrier': 'FedEx', 'rate': '45.05', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '45.05', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_19c859344a4a4f17a8997c9ae8ff9f8f', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FEDEX_GROUND', 'carrier': 'FedEx', 'rate': '17.27', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '17.27', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 2, 'delivery_date': '2022-09-16T23:59:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 2, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}, {'id': 'rate_7afecdaa1a924b92837827f95e4e70d6', 'object': 'Rate', 'created_at': None, 'updated_at': None, 'mode': 'test', 'service': 'FEDEX_EXPRESS_SAVER', 'carrier': 'FedEx', 'rate': '40.63', 'currency': 'USD', 'retail_rate': None, 'retail_currency': None, 'list_rate': '40.63', 'list_currency': 'USD', 'billing_type': 'carrier', 'delivery_days': 5, 'delivery_date': '2022-09-19T16:30:00Z', 'delivery_date_guaranteed': True, 'est_delivery_days': 5, 'shipment_id': 'shp_13e869f50e7e4ffdb4e4a86e2a7a9d34', 'carrier_account_id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15'}], 'id': 'order_7bf14e4a887c42d7b41f06a79bdf13ea', 'object': 'Order'},
                    'shipments/shp_13e869f50e7e4ffdb4e4a86e2a7a9d34/insure': {'error': {'code': 'INSURANCE.AMOUNT.BELOW_MINIMUM', 'message': 'Insurance amount must be greater than 0', 'errors': []}},
                },
                'get': {
                    'carrier_accounts': [{'id': 'ca_2c2e2fdebde54bb6a006bf1642b92f15', 'object': 'CarrierAccount', 'type': 'FedexAccount', 'clone': False, 'created_at': '2017-06-08T06:02:05Z', 'updated_at': '2022-08-25T22:43:01Z', 'description': 'FedEx Account', 'reference': 'FedEx for test', 'billing_type': 'carrier', 'readable': 'FedEx', 'logo': None, 'fields': {'credentials': {'account_number': {'visibility': 'readonly', 'label': 'FedEx Account Number', 'value': ''}}, 'test_credentials': {'account_number': {'visibility': 'readonly', 'label': 'FedEx Account Number', 'value': '601356805'}}}, 'credentials': {'account_number': ''}, 'test_credentials': {'account_number': '601356805'}}, {'id': 'ca_0df4ffb8af0c42a29fb5338a728bc94a', 'object': 'CarrierAccount', 'type': 'DhlGlobalMailAccount', 'clone': False, 'created_at': '2017-07-04T07:41:49Z', 'updated_at': '2018-11-26T06:39:23Z', 'description': 'DHL GLOBAL', 'reference': 'TEST', 'billing_type': 'carrier', 'readable': 'DHL Global Mail', 'logo': None, 'fields': {'credentials': {'account_number': {'visibility': 'visible', 'label': 'DHL Global Mail Pickup Number', 'value': ''}, 'username': {'visibility': 'visible', 'label': 'DHL Global Mail Username', 'value': '803921577'}, 'password': {'visibility': 'password', 'label': 'DHL Global Mail Password', 'value': '********'}, 'client_id': {'visibility': 'visible', 'label': 'DHL Global Mail ClientID', 'value': 'CustomerTest'}, 'facility_code': {'visibility': 'visible', 'label': 'DHL Global Mail Facility Code', 'value': 'AM'}, 'mailer_id': {'visibility': 'visible', 'label': 'DHL Global Mail Mailer ID', 'value': ''}, 'return_location': {'visibility': 'visible', 'label': 'DHL Global Mail Return Location ID', 'value': ''}, 'ftp_username': {'visibility': 'visible', 'label': 'FTP User ID', 'value': ''}, 'ftp_password': {'visibility': 'password', 'label': 'FTP Password', 'value': ''}}}, 'credentials': {'account_number': '', 'username': '803921577', 'password': '********', 'client_id': 'CustomerTest', 'facility_code': 'AM', 'mailer_id': '', 'return_location': '', 'ftp_username': '', 'ftp_password': ''}, 'test_credentials': None}, {'id': 'ca_aa8a712208644e7e99e0d88bf2bca0b9', 'object': 'CarrierAccount', 'type': 'UspsAccount', 'clone': False, 'created_at': '2020-12-14T12:18:23Z', 'updated_at': '2021-01-07T19:44:47Z', 'description': 'USPS Account', 'reference': None, 'billing_type': 'easypost', 'readable': 'USPS', 'logo': None, 'fields': {'credentials': {'company_name': {'visibility': 'visible', 'label': 'Account Holder Name', 'value': 'testodoo'}, 'address_street': {'visibility': 'visible', 'label': 'Street Address', 'value': '1033 Elm Hill Pike'}, 'address_city': {'visibility': 'visible', 'label': 'City', 'value': 'Nashville'}, 'address_state': {'visibility': 'visible', 'label': 'State', 'value': 'TN'}, 'address_zip': {'visibility': 'visible', 'label': 'ZIP Code', 'value': '37210'}, 'phone': {'visibility': 'visible', 'label': 'Phone Number', 'value': None}, 'email': {'visibility': 'visible', 'label': 'Email Address', 'value': 'lma@odoo.com'}, 'shipper_id': {'visibility': 'visible', 'label': 'Shipper Identifier (Optional)', 'value': None}}}, 'credentials': {'company_name': 'testodoo', 'address_street': '1033 Elm Hill Pike', 'address_city': 'Nashville', 'address_state': 'TN', 'address_zip': '37210', 'phone': None, 'email': 'lma@odoo.com', 'shipper_id': None}, 'test_credentials': None}, {'id': 'ca_ecae37f6b8f14faf9753086356ce811c', 'object': 'CarrierAccount', 'type': 'GsoAccount', 'clone': False, 'created_at': '2021-04-23T06:29:33Z', 'updated_at': '2021-04-28T08:49:09Z', 'description': 'GSO Account', 'reference': None, 'billing_type': 'carrier', 'readable': 'GSO', 'logo': None, 'fields': {'credentials': {'account_number': {'visibility': 'visible', 'label': 'GSO Account Number', 'value': '2503982901'}, 'username': {'visibility': 'visible', 'label': 'Username', 'value': 'dut@odoo.com'}, 'password': {'visibility': 'password', 'label': 'Password', 'value': '********'}}}, 'credentials': {'account_number': '2503982901', 'username': 'dut@odoo.com', 'password': '********'}, 'test_credentials': None}, {'id': 'ca_c33bfc2de51142858b254c2c4cb9898c', 'object': 'CarrierAccount', 'type': 'DhlExpressAccount', 'clone': False, 'created_at': '2021-05-21T15:53:31Z', 'updated_at': '2022-02-04T08:34:13Z', 'description': 'DHL Express Account', 'reference': None, 'billing_type': 'carrier', 'readable': 'DHL Express', 'logo': None, 'fields': {'credentials': {'account_number': {'visibility': 'visible', 'label': 'DHL Account Number', 'value': '967395784'}, 'country': {'visibility': 'visible', 'label': 'Account Country Code (2 Letter)', 'value': 'BE'}, 'site_id': {'visibility': 'visible', 'label': 'Site ID (Optional)', 'value': 'v62_X4e7G4Ww0y'}, 'password': {'visibility': 'password', 'label': 'Password (Optional)', 'value': '********'}, 'is_reseller': {'visibility': 'checkbox', 'label': 'Reseller Account? (check if yes)', 'value': None}}}, 'credentials': {'account_number': '967395784', 'country': 'BE', 'site_id': 'v62_X4e7G4Ww0y', 'password': '********', 'is_reseller': None}, 'test_credentials': None}, {'id': 'ca_40c3b684c86e451aa4518f35ba13f373', 'object': 'CarrierAccount', 'type': 'CanadaPostAccount', 'clone': False, 'created_at': '2022-01-25T15:52:11Z', 'updated_at': '2022-06-01T21:14:59Z', 'description': 'Canada Post Account', 'reference': None, 'billing_type': 'carrier', 'readable': 'Canada Post', 'logo': None, 'fields': {'credentials': {'api_key': {'visibility': 'masked', 'label': 'CanadaPost API Key', 'value': '2f7c***********************************'}, 'contract_id': {'visibility': 'visible', 'label': 'CanadaPost Contract ID (contract customers only)', 'value': '0043862026'}, 'customer_number': {'visibility': 'visible', 'label': 'CanadaPost Customer Number', 'value': '0008476014'}, 'has_credit_card': {'visibility': 'checkbox', 'label': 'CanadaPost account has credit card?', 'value': 'true'}}, 'test_credentials': {'api_key': {'visibility': 'masked', 'label': 'CanadaPost Test API Key', 'value': ''}, 'contract_id': {'visibility': 'visible', 'label': 'CanadaPost Test Contract ID (contract customers only)', 'value': ''}, 'customer_number': {'visibility': 'visible', 'label': 'CanadaPost Test Customer Number', 'value': ''}, 'has_credit_card': {'visibility': 'checkbox', 'label': 'CanadaPost Test account has credit card?', 'value': None}}}, 'credentials': {'api_key': '2f7c***********************************', 'contract_id': '0043862026', 'customer_number': '0008476014', 'has_credit_card': 'true'}, 'test_credentials': {'api_key': '', 'contract_id': '', 'customer_number': '', 'has_credit_card': None}}, {'id': 'ca_eb3e44d1985b4075aed1ded815db19f1', 'object': 'CarrierAccount', 'type': 'CanparAccount', 'clone': False, 'created_at': '2022-04-06T11:38:48Z', 'updated_at': '2022-04-06T11:38:48Z', 'description': 'Canpar Account', 'reference': None, 'billing_type': 'carrier', 'readable': 'Canpar', 'logo': None, 'fields': {'credentials': {'account_number': {'visibility': 'visible', 'label': 'Canpar Account Number', 'value': None}, 'username': {'visibility': 'visible', 'label': 'Canpar Username', 'value': None}, 'password': {'visibility': 'password', 'label': 'Canpar Password', 'value': ''}}, 'test_credentials': {'account_number': {'visibility': 'visible', 'label': 'Test Canpar Account Number', 'value': None}, 'username': {'visibility': 'visible', 'label': 'Test Canpar Username', 'value': None}, 'password': {'visibility': 'password', 'label': 'Test Canpar Password', 'value': ''}}}, 'credentials': {'account_number': None, 'username': None, 'password': ''}, 'test_credentials': {'account_number': None, 'username': None, 'password': ''}}, {'id': 'ca_880d74674a9645b1b986423122de37cc', 'object': 'CarrierAccount', 'type': 'DpdAccount', 'clone': False, 'created_at': '2022-05-09T09:11:44Z', 'updated_at': '2022-05-09T09:22:58Z', 'description': 'Test', 'reference': 'DPD MCB', 'billing_type': 'carrier', 'readable': 'DPD', 'logo': None, 'fields': {'credentials': {'username': {'visibility': 'visible', 'label': 'WebConnect Username', 'value': 'dut@odoo.com'}, 'password': {'visibility': 'password', 'label': 'WebConnect Password', 'value': '********'}, 'customer_number': {'visibility': 'visible', 'label': 'DPD Customer Number', 'value': None}}, 'test_credentials': {'username': {'visibility': 'visible', 'label': 'Test WebConnect Username', 'value': 'dut@odoo.com'}, 'password': {'visibility': 'password', 'label': 'Test WebConnect Password', 'value': '********'}, 'customer_number': {'visibility': 'visible', 'label': 'DPD Customer Number', 'value': None}}}, 'credentials': {'username': 'dut@odoo.com', 'password': '********', 'customer_number': None}, 'test_credentials': {'username': 'dut@odoo.com', 'password': '********', 'customer_number': None}}, {'id': 'ca_e50455de32ee4263be60184b597fdabd', 'object': 'CarrierAccount', 'type': 'LsoDefaultAccount', 'clone': False, 'created_at': '2022-05-18T17:30:29Z', 'updated_at': '2022-05-18T17:30:29Z', 'description': 'LSO Account', 'reference': None, 'billing_type': 'easypost', 'readable': 'LSO', 'logo': None, 'fields': {'credentials': {}}, 'credentials': {}, 'test_credentials': None}, {'id': 'ca_6be2f8d486f14366846b3382bdf0aff9', 'object': 'CarrierAccount', 'type': 'UpsDapAccount', 'clone': False, 'created_at': '2022-05-23T16:34:11Z', 'updated_at': '2022-05-23T16:34:11Z', 'description': 'UPS Digital Access Program Account', 'reference': None, 'billing_type': 'easypost', 'readable': 'UPSDAP', 'logo': None, 'fields': {'credentials': {'account_number': {'visibility': 'readonly', 'label': 'UPS Account Number', 'value': None}}}, 'credentials': {'account_number': None}, 'test_credentials': None}],
                },
            }
            response = mocked_contents.get(request_type, {}).get(endpoint)
            if response is None:
                raise NotImplementedError('Easypost mock request not implemented for this method/endpoint: %s %s' % (request_type, endpoint))
            return response

        def _mocked_easypost_get_services_and_package_types():
            """ In the module, we use requests to access those documents. """
            with file_open('delivery_easypost/static/data/package_types_by_carriers.json') as f:
                packages = json.load(f)
            with file_open('delivery_easypost/static/data/services_by_carriers.json') as f:
                services = json.load(f)
            return packages, services

        def _mocked_successful_empty_get_response(*args, **kwargs):
            """ This mock is used when requesting documents, such as labels. """
            response = Mock()
            response.status_code = 200
            response.content = ''
            return response

        with patch.object(DeliverCarrier, '_easypost_get_services_and_package_types', side_effect=_mocked_easypost_get_services_and_package_types), \
             patch.object(EasypostRequest, '_make_api_request', side_effect=_mocked_easypost_request), \
             patch.object(requests.Session, 'get', _mocked_successful_empty_get_response):
            yield
