﻿Imports System.IO
Imports System.Net.Http
Imports System.Text
Imports Newtonsoft.Json

Public Class Form1


    Private FrmV_LawsFolder As String = "Data\Laws"
    Private Const FrmV_AI_URL As String = "http://localhost:11434/api/generate"
    Private Const FrmV_FAST_MODEL As String = "mistral"
    Private Const FrmV_SMART_MODEL As String = "mistral"
    Private Const FrmV_LAW_LIMIT As Integer = 1500

    Private FrmV_lawIndex As New Dictionary(Of String, String)
    Private FrmV_aiCache As New Dictionary(Of String, String)
    Private Shared FrmO_httpClient As New HttpClient()
    Private Const FrmV_MAX_CACHE As Integer = 50

    Private Sub Form1_Load(sender As Object, e As EventArgs) Handles MyBase.Load
        Me.Text = "Legal AI Assistant"
        FrmO_httpClient.Timeout = TimeSpan.FromMinutes(2)

        FrmV_LawsFolder = My.Settings.LawsFolder
        If String.IsNullOrEmpty(FrmV_LawsFolder) Then
            FrmV_LawsFolder = "Data\Laws"
        End If

        Txt_LawsPath.Text = FrmV_LawsFolder
        LoadLawIndex()

        lblSelectMeaning.Visible = False
        Lst_Interpretations.Visible = False
        Btn_ConfirmInterpretation.Visible = False
    End Sub

    Private Sub Btn_BrowseFolder_Click(sender As Object, e As EventArgs) Handles Btn_BrowseFolder.Click
        Using folderDialog As New FolderBrowserDialog()
            folderDialog.Description = "Select the folder containing your law files (.txt)"
            folderDialog.SelectedPath = FrmV_LawsFolder
            folderDialog.ShowNewFolderButton = True

            If folderDialog.ShowDialog() = DialogResult.OK Then
                FrmV_LawsFolder = folderDialog.SelectedPath
                Txt_LawsPath.Text = FrmV_LawsFolder

                My.Settings.LawsFolder = FrmV_LawsFolder
                My.Settings.Save()

                FrmV_lawIndex.Clear()
                LoadLawIndex()

                MessageBox.Show($"Folder updated!{vbCrLf}{FrmV_lawIndex.Count} laws loaded.",
                              "Folder Changed", MessageBoxButtons.OK, MessageBoxIcon.Information)
            End If
        End Using
    End Sub

    Private Sub LoadLawIndex()
        Try
            If Not Directory.Exists(FrmV_LawsFolder) Then
                Directory.CreateDirectory(FrmV_LawsFolder)
                Me.Text = "Legal AI Assistant (0 laws loaded)"
                Return
            End If

            Dim files = Directory.GetFiles(FrmV_LawsFolder, "*.txt")
            Dim count As Integer = 0

            If files.Length > 10 Then
                Me.Text = "Loading laws..."
                Application.DoEvents()
            End If

            For Each file In files
                Try
                    Dim fileName = Path.GetFileName(file).ToLower()
                    FrmV_lawIndex(fileName) = IO.File.ReadAllText(file, Encoding.UTF8)
                    count += 1

                    If count Mod 10 = 0 AndAlso files.Length > 10 Then
                        Me.Text = $"Loading laws... {count}/{files.Length}"
                        Application.DoEvents()
                    End If
                Catch

                End Try
            Next

            If count > 0 Then
                Me.Text = $"Legal AI Assistant ({count} laws loaded)"
            Else
                Me.Text = "Legal AI Assistant (No laws found)"
                MessageBox.Show($"No .txt files found in:{vbCrLf}{FrmV_LawsFolder}{vbCrLf}{vbCrLf}" &
                              "Please add law files or select a different folder.",
                              "No Laws Found", MessageBoxButtons.OK, MessageBoxIcon.Information)
            End If
        Catch ex As Exception
            MessageBox.Show("Warning: " & ex.Message, "Warning",
                          MessageBoxButtons.OK, MessageBoxIcon.Warning)
        End Try
    End Sub

    Private Async Sub Btn_AskAI_Click(sender As Object, e As EventArgs) Handles Btn_AskAI.Click

        If Txt_Question.Text.Trim() = "" Then
            MessageBox.Show("Please enter a question", "Input Required",
                          MessageBoxButtons.OK, MessageBoxIcon.Information)
            Return
        End If

        If FrmV_lawIndex.Count = 0 Then
            MessageBox.Show("No laws loaded. Please select a folder with law files.",
                          "No Laws Available", MessageBoxButtons.OK, MessageBoxIcon.Warning)
            Return
        End If

        Btn_AskAI.Enabled = False
        Btn_AskAI.Text = "Analyzing..."
        Txt_Answer.Clear()
        Lst_Interpretations.Items.Clear()
        Application.DoEvents()

        Try
            Dim interpretations = Await GetInterpretations(Txt_Question.Text)

            If interpretations.Count = 0 Then
                MessageBox.Show("Could not generate interpretations. Please try again.",
                              "Error", MessageBoxButtons.OK, MessageBoxIcon.Warning)
                Return
            End If

            For Each i In interpretations
                Lst_Interpretations.Items.Add(i)
            Next

            If Lst_Interpretations.Items.Count > 0 Then
                Lst_Interpretations.SelectedIndex = 0
            End If

            lblSelectMeaning.Visible = True
            Lst_Interpretations.Visible = True
            Btn_ConfirmInterpretation.Visible = True

        Catch ex As Exception
            MessageBox.Show("Error: " & ex.Message, "Error",
                          MessageBoxButtons.OK, MessageBoxIcon.Error)
        Finally
            Btn_AskAI.Text = "Analyze Question"
            Btn_AskAI.Enabled = True
        End Try

    End Sub

    Private Async Sub Btn_ConfirmInterpretation_Click(sender As Object, e As EventArgs) _
        Handles Btn_ConfirmInterpretation.Click

        If Lst_Interpretations.SelectedItem Is Nothing Then
            MessageBox.Show("Please select an interpretation", "Selection Required",
                          MessageBoxButtons.OK, MessageBoxIcon.Information)
            Return
        End If

        Dim selectedMeaning As String = Lst_Interpretations.SelectedItem.ToString()

        Btn_ConfirmInterpretation.Enabled = False
        Btn_ConfirmInterpretation.Text = "Processing..."
        Txt_Answer.Text = "Extracting legal keywords and searching laws..."
        Application.DoEvents()

        Try

            Dim keywords = Await ExtractKeywords(selectedMeaning)

            If keywords.Count = 0 Then
                Txt_Answer.Text = "Could not identify legal concepts. Please rephrase."
                Return
            End If

            Dim lawText = FindRelevantLaw(keywords)

            If lawText = "" Then
                Txt_Answer.Text = $"No relevant law found.{vbCrLf}{vbCrLf}" &
                                $"Searched for keywords: {String.Join(", ", keywords)}{vbCrLf}{vbCrLf}" &
                                $"Available laws (top 10): {String.Join(", ", FrmV_lawIndex.Keys.Take(10))}"
                Return
            End If

            ' Get AI explanation
            Dim prompt = BuildExplanationPrompt(lawText, selectedMeaning)
            Txt_Answer.Text = Await CallAI(prompt, FrmV_SMART_MODEL)

            ' Hide interpretation UI
            lblSelectMeaning.Visible = False
            Lst_Interpretations.Visible = False
            Btn_ConfirmInterpretation.Visible = False

        Catch ex As Exception
            Txt_Answer.Text = "Error: " & ex.Message
            MessageBox.Show("Error: " & ex.Message, "Error",
                          MessageBoxButtons.OK, MessageBoxIcon.Error)
        Finally
            Btn_ConfirmInterpretation.Text = "Confirm Interpretation"
            Btn_ConfirmInterpretation.Enabled = True
        End Try

    End Sub

    Private Async Function GetInterpretations(q As String) As Task(Of List(Of String))
        Dim prompt As String =
        "You are a legal analyst. Analyze the following question and provide exactly 3 distinct legal interpretations." & vbCrLf &
        "Question: " & q & vbCrLf &
        "Return ONLY the 3 interpretations, one per line. No introduction, no numbers."

        Dim response = Await CallAI(prompt, FrmV_FAST_MODEL)

        Dim lines = response.Split({vbCrLf, vbLf}, StringSplitOptions.RemoveEmptyEntries)
        Dim cleanedList As New List(Of String)

        For Each line In lines
            Dim cleaned = System.Text.RegularExpressions.Regex.Replace(line.Trim(), "^[\d\.\-\*\)]+\s*", "")
            If cleaned.Length > 3 Then
                cleanedList.Add(cleaned)
            End If
        Next

        Return cleanedList.Take(3).ToList()
    End Function

    Private Async Function ExtractKeywords(text As String) As Task(Of List(Of String))
        Dim prompt = "Extract 5-7 important legal keywords from this text as a comma-separated list:" & vbCrLf & text
        Dim response = Await CallAI(prompt, FrmV_FAST_MODEL)

        Return response.Split({","c, vbCrLf, vbLf}, StringSplitOptions.RemoveEmptyEntries).
            Select(Function(x) x.Trim().ToLower()).
            Where(Function(x) x.Length > 2).
            Distinct().
            Take(10).
            ToList()
    End Function

    Private Function FindRelevantLaw(keywords As List(Of String)) As String
        If FrmV_lawIndex.Count = 0 Then
            Return ""
        End If

        Dim results As New List(Of Tuple(Of String, String, Integer))

        For Each kv In FrmV_lawIndex
            Dim fileName = kv.Key.ToLower()
            Dim content = kv.Value.ToLower()
            Dim score As Integer = 0

            For Each keyword In keywords
                If fileName.Contains(keyword) Then
                    score += 10
                End If
            Next

            For Each keyword In keywords
                If content.Contains(keyword) Then
                    score += 1
                End If
            Next

            If score > 0 Then
                results.Add(New Tuple(Of String, String, Integer)(kv.Key, kv.Value, score))
            End If
        Next

        results = results.OrderByDescending(Function(r) r.Item3).ToList()

        Dim sb As New StringBuilder()
        For Each result In results.Take(2)
            sb.AppendLine($"SOURCE: {result.Item1.Replace(".txt", "")} (Relevance: {result.Item3})")
            sb.AppendLine(result.Item2)
            sb.AppendLine("----------------------------------")
        Next

        Return sb.ToString()
    End Function

    Private Function BuildExplanationPrompt(law As String, q As String) As String
        Dim safeLaw = If(law.Length > FrmV_LAW_LIMIT,
                        law.Substring(0, FrmV_LAW_LIMIT) & "...",
                        law)

        Return "You are a legal assistant. Answer this question using ONLY the law text provided below." & vbCrLf & vbCrLf &
               "LAW TEXT:" & vbCrLf &
               safeLaw & vbCrLf & vbCrLf &
               "QUESTION: " & q & vbCrLf & vbCrLf &
               "Provide a clear, helpful answer based solely on the law text above:"
    End Function

    Private Async Function CallAI(prompt As String, model As String) As Task(Of String)
        Dim key = model & "::" & prompt.GetHashCode()
        If FrmV_aiCache.ContainsKey(key) Then
            Return FrmV_aiCache(key)
        End If

        Dim payload = New With {
            .model = model,
            .prompt = prompt,
            .stream = False,
            .options = New With {
                .temperature = 0.3,
                .num_predict = 400,
                .top_k = 20,
                .top_p = 0.9
            }
        }

        Dim json = JsonConvert.SerializeObject(payload)
        Dim content As New StringContent(json, Encoding.UTF8, "application/json")
        Dim resp = Await FrmO_httpClient.PostAsync(FrmV_AI_URL, content)

        If Not resp.IsSuccessStatusCode Then
            Throw New Exception(Await resp.Content.ReadAsStringAsync())
        End If

        Dim txt = Await resp.Content.ReadAsStringAsync()
        Dim obj = JsonConvert.DeserializeObject(Of Dictionary(Of String, Object))(txt)

        If Not obj.ContainsKey("response") Then
            Throw New Exception("Invalid AI response")
        End If

        Dim answer = obj("response").ToString().Trim()

        If FrmV_aiCache.Count >= FrmV_MAX_CACHE Then
            FrmV_aiCache.Remove(FrmV_aiCache.Keys.First())
        End If
        FrmV_aiCache(key) = answer

        Return answer
    End Function

    Private Sub Form1_KeyDown(sender As Object, e As KeyEventArgs) Handles MyBase.KeyDown
        If e.Control AndAlso e.KeyCode = Keys.R Then
            FrmV_aiCache.Clear()
            MessageBox.Show($"Cache cleared!{vbCrLf}{FrmV_lawIndex.Count} laws loaded.",
                          "Cache Cleared", MessageBoxButtons.OK, MessageBoxIcon.Information)
        End If
    End Sub

End Class