﻿Imports System.IO
Imports System.Drawing
Imports System.Drawing.Imaging
Imports DocumentFormat.OpenXml.Packaging

Public Class Viewer
    Inherits System.Web.UI.Page


    Private Property FilePath As String
        Get
            Return If(ViewState("FilePath"), String.Empty).ToString()
        End Get
        Set(value As String)
            ViewState("FilePath") = value
        End Set
    End Property

    Private Property CurrentPage As Integer
        Get
            Return If(ViewState("CurrentPage"), 0)
        End Get
        Set(value As Integer)
            ViewState("CurrentPage") = value
        End Set
    End Property

    Private Property TotalPages As Integer
        Get
            Return If(ViewState("TotalPages"), 0)
        End Get
        Set(value As Integer)
            ViewState("TotalPages") = value
        End Set
    End Property

    Private Property ZoomLevel As Integer
        Get
            Return If(ViewState("ZoomLevel"), 100)
        End Get
        Set(value As Integer)
            ViewState("ZoomLevel") = value
        End Set
    End Property

    Private Property FileType As String
        Get
            Return If(ViewState("FileType"), String.Empty).ToString()
        End Get
        Set(value As String)
            ViewState("FileType") = value
        End Set
    End Property

    Protected Sub Page_Load(sender As Object, e As EventArgs) Handles Me.Load
        If Not IsPostBack Then
            Dim filePath As String = Request.QueryString("file")
            Dim pageNum As String = Request.QueryString("page")


            If String.IsNullOrEmpty(filePath) Then
                ShowError("No file path provided.")
                Return
            End If

            If Not File.Exists(filePath) Then
                ShowError("File not found: " & filePath)
                Return
            End If


            Dim fullPath As String = Path.GetFullPath(filePath)
            Dim allowedPath As String = Path.GetFullPath(LuceneSearch.ALLOWED_ROOT)

            If Not fullPath.StartsWith(allowedPath, StringComparison.OrdinalIgnoreCase) Then
                ShowError("Access denied. File is outside allowed directory.")
                Return
            End If


            Me.FilePath = filePath


            Dim extension As String = Path.GetExtension(filePath).ToLower()
            Me.FileType = extension


            Dim requestedPage As Integer = 0
            If Not String.IsNullOrEmpty(pageNum) Then
                Integer.TryParse(pageNum, requestedPage)
            End If


            Select Case extension
                Case ".tif", ".tiff"
                    InitializeTiffViewer(requestedPage)
                Case ".txt"
                    InitializeTextViewer()
                Case ".docx"
                    InitializeDocxViewer()
                Case Else
                    ShowError("Unsupported file type: " & extension & ". This viewer supports .txt, .docx, .tif, and .tiff files.")
            End Select
        End If
    End Sub

    Private Sub InitializeTextViewer()
        Try

            lblFileName.Text = Path.GetFileName(FilePath)
            lblPageInfo.Text = "Text Document"


            Dim content As String = File.ReadAllText(FilePath)


            DisplayTextAsImage(content)


            pnlNavigation.Visible = False
            pnlImage.Visible = True

        Catch ex As Exception
            ShowError("Error loading text file: " & ex.Message)
        End Try
    End Sub

    Private Sub InitializeDocxViewer()
        Try

            lblFileName.Text = Path.GetFileName(FilePath)
            lblPageInfo.Text = "Word Document"


            Dim content As String = ""
            Using doc As WordprocessingDocument = WordprocessingDocument.Open(FilePath, False)
                content = doc.MainDocumentPart.Document.Body.InnerText
            End Using

            DisplayTextAsImage(content)

            pnlNavigation.Visible = False
            pnlImage.Visible = True

        Catch ex As Exception
            ShowError("Error loading Word document: " & ex.Message)
        End Try
    End Sub

    Private Sub DisplayTextAsImage(text As String)

        Dim width As Integer = 800
        Dim font As New Font("Courier New", 10)


        Using measureBitmap As New Bitmap(1, 1)
            Using g As Graphics = Graphics.FromImage(measureBitmap)
                Dim textSize As SizeF = g.MeasureString(text, font, width - 40)
                Dim height As Integer = CInt(textSize.Height) + 40

                Using bitmap As New Bitmap(width, Math.Max(height, 600))
                    Using g2 As Graphics = Graphics.FromImage(bitmap)

                        g2.Clear(Color.White)

                        g2.DrawString(text, font, Brushes.Black, New RectangleF(20, 20, width - 40, height - 40))
                    End Using

                    Using ms As New MemoryStream()
                        bitmap.Save(ms, ImageFormat.Png)
                        imgViewer.ImageUrl = "data:image/png;base64," & Convert.ToBase64String(ms.ToArray())
                    End Using
                End Using
            End Using
        End Using
    End Sub

    Private Sub InitializeTiffViewer(requestedPage As Integer)
        Try

            lblFileName.Text = Path.GetFileName(FilePath)


            Using img As Image = Image.FromFile(FilePath)
                Dim dimension As New FrameDimension(img.FrameDimensionsList(0))
                Me.TotalPages = img.GetFrameCount(dimension)
            End Using

            If requestedPage < 0 Then requestedPage = 0
            If requestedPage >= TotalPages Then requestedPage = TotalPages - 1

            Me.CurrentPage = requestedPage

            DisplayTiffPage()


            UpdateNavigationControls()

        Catch ex As Exception
            ShowError("Error loading TIFF file: " & ex.Message)
        End Try
    End Sub

    Private Sub DisplayTiffPage()
        Try
            Using img As Image = Image.FromFile(FilePath)

                Dim dimension As New FrameDimension(img.FrameDimensionsList(0))
                img.SelectActiveFrame(dimension, CurrentPage)


                Dim zoomFactor As Single = ZoomLevel / 100.0F
                Dim newWidth As Integer = CInt(img.Width * zoomFactor)
                Dim newHeight As Integer = CInt(img.Height * zoomFactor)

                Using resizedImg As New Bitmap(newWidth, newHeight)
                    Using g As Graphics = Graphics.FromImage(resizedImg)
                        g.InterpolationMode = Drawing2D.InterpolationMode.HighQualityBicubic
                        g.DrawImage(img, 0, 0, newWidth, newHeight)
                    End Using


                    Using ms As New MemoryStream()
                        resizedImg.Save(ms, ImageFormat.Png)
                        imgViewer.ImageUrl = "data:image/png;base64," & Convert.ToBase64String(ms.ToArray())
                    End Using
                End Using
            End Using

            pnlImage.Visible = True
            pnlError.Visible = False

        Catch ex As Exception
            ShowError("Error displaying page: " & ex.Message)
        End Try
    End Sub

    Private Sub UpdateNavigationControls()

        pnlNavigation.Visible = (TotalPages > 1)

        If pnlNavigation.Visible Then

            lblPageInfo.Text = String.Format("Page {0} of {1}", CurrentPage + 1, TotalPages)
            lblTotalPages.Text = TotalPages.ToString()
            txtPageNumber.Text = (CurrentPage + 1).ToString()

            btnFirst.Enabled = (CurrentPage > 0)
            btnPrevious.Enabled = (CurrentPage > 0)
            btnNext.Enabled = (CurrentPage < TotalPages - 1)
            btnLast.Enabled = (CurrentPage < TotalPages - 1)

            lblZoom.Text = ZoomLevel.ToString() & "%"
        End If
    End Sub

    Protected Sub btnFirst_Click(sender As Object, e As EventArgs)
        CurrentPage = 0
        DisplayTiffPage()
        UpdateNavigationControls()
    End Sub

    Protected Sub btnPrevious_Click(sender As Object, e As EventArgs)
        If CurrentPage > 0 Then
            CurrentPage -= 1
            DisplayTiffPage()
            UpdateNavigationControls()
        End If
    End Sub

    Protected Sub btnNext_Click(sender As Object, e As EventArgs)
        If CurrentPage < TotalPages - 1 Then
            CurrentPage += 1
            DisplayTiffPage()
            UpdateNavigationControls()
        End If
    End Sub

    Protected Sub btnLast_Click(sender As Object, e As EventArgs)
        CurrentPage = TotalPages - 1
        DisplayTiffPage()
        UpdateNavigationControls()
    End Sub

    Protected Sub txtPageNumber_TextChanged(sender As Object, e As EventArgs)
        Dim pageNum As Integer
        If Integer.TryParse(txtPageNumber.Text, pageNum) Then
            pageNum -= 1
            If pageNum >= 0 AndAlso pageNum < TotalPages Then
                CurrentPage = pageNum
                DisplayTiffPage()
                UpdateNavigationControls()
            Else
                txtPageNumber.Text = (CurrentPage + 1).ToString()
            End If
        Else
            txtPageNumber.Text = (CurrentPage + 1).ToString()
        End If
    End Sub

    Protected Sub btnZoomIn_Click(sender As Object, e As EventArgs)
        If ZoomLevel < 200 Then
            ZoomLevel += 25
            If FileType = ".tif" OrElse FileType = ".tiff" Then
                DisplayTiffPage()
            End If
            UpdateNavigationControls()
        End If
    End Sub

    Protected Sub btnZoomOut_Click(sender As Object, e As EventArgs)
        If ZoomLevel > 25 Then
            ZoomLevel -= 25
            If FileType = ".tif" OrElse FileType = ".tiff" Then
                DisplayTiffPage()
            End If
            UpdateNavigationControls()
        End If
    End Sub

    Protected Sub btnZoomReset_Click(sender As Object, e As EventArgs)
        ZoomLevel = 100
        If FileType = ".tif" OrElse FileType = ".tiff" Then
            DisplayTiffPage()
        End If
        UpdateNavigationControls()
    End Sub

    Private Sub ShowError(message As String)
        lblError.Text = message
        pnlError.Visible = True
        pnlImage.Visible = False
        pnlNavigation.Visible = False
    End Sub
End Class